// Spotify-specific JavaScript functionality

class SpotifyApp {
    constructor() {
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.setupFormValidation();
        this.setupAnimations();
    }

    setupEventListeners() {
        // Social login buttons
        document.querySelectorAll('.spotify-button-secondary').forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                this.handleSocialLogin(e.target);
            });
        });

        // Form submissions
        document.querySelectorAll('form').forEach(form => {
            form.addEventListener('submit', (e) => {
                this.handleFormSubmit(e);
            });
        });

        // Input focus effects
        document.querySelectorAll('.spotify-input').forEach(input => {
            input.addEventListener('focus', (e) => {
                this.handleInputFocus(e.target);
            });
            
            input.addEventListener('blur', (e) => {
                this.handleInputBlur(e.target);
            });
        });
    }

    setupFormValidation() {
        // Email validation
        const emailInputs = document.querySelectorAll('input[type="email"]');
        emailInputs.forEach(input => {
            input.addEventListener('input', (e) => {
                this.validateEmail(e.target);
            });
        });

        // Password validation
        const passwordInputs = document.querySelectorAll('input[type="password"]');
        passwordInputs.forEach(input => {
            input.addEventListener('input', (e) => {
                this.validatePassword(e.target);
            });
        });
    }

    setupAnimations() {
        // Add fade-in animation to cards
        document.querySelectorAll('.spotify-card').forEach((card, index) => {
            card.style.animationDelay = `${index * 0.1}s`;
            card.classList.add('fade-in');
        });

        // Add hover effects to buttons
        document.querySelectorAll('.spotify-button, .spotify-button-secondary').forEach(button => {
            button.addEventListener('mouseenter', () => {
                button.style.transform = 'scale(1.04)';
            });
            
            button.addEventListener('mouseleave', () => {
                button.style.transform = 'scale(1)';
            });
        });
    }

    handleSocialLogin(button) {
        const provider = button.textContent.trim().toLowerCase();
        
        // Show loading state
        const originalText = button.textContent;
        button.textContent = 'Connecting...';
        button.disabled = true;

        // Simulate API call
        setTimeout(() => {
            button.textContent = originalText;
            button.disabled = false;
            
            // Show success message
            this.showNotification(`Connected with ${provider}`, 'success');
        }, 2000);
    }

    handleFormSubmit(e) {
        const form = e.target;
        const submitButton = form.querySelector('button[type="submit"]');
        
        if (submitButton) {
            // Show loading state
            const originalText = submitButton.textContent;
            submitButton.textContent = 'Processing...';
            submitButton.disabled = true;
            
            // Add loading spinner
            submitButton.innerHTML = '<div class="spotify-spinner"></div> Processing...';
        }

        // Validate form
        if (!this.validateForm(form)) {
            e.preventDefault();
            if (submitButton) {
                submitButton.textContent = originalText;
                submitButton.disabled = false;
            }
            return false;
        }

        // Show loading overlay
        this.showLoadingOverlay();
    }

    handleInputFocus(input) {
        input.parentElement.classList.add('focused');
    }

    handleInputBlur(input) {
        input.parentElement.classList.remove('focused');
    }

    validateEmail(input) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        const isValid = emailRegex.test(input.value);
        
        this.updateInputState(input, isValid);
        return isValid;
    }

    validatePassword(input) {
        const isValid = input.value.length >= 6;
        
        this.updateInputState(input, isValid);
        return isValid;
    }

    validateForm(form) {
        let isValid = true;
        
        // Validate required fields
        const requiredFields = form.querySelectorAll('[required]');
        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                this.showFieldError(field, 'This field is required');
                isValid = false;
            } else {
                this.clearFieldError(field);
            }
        });

        // Validate email fields
        const emailFields = form.querySelectorAll('input[type="email"]');
        emailFields.forEach(field => {
            if (field.value && !this.validateEmail(field)) {
                isValid = false;
            }
        });

        // Validate password fields
        const passwordFields = form.querySelectorAll('input[type="password"]');
        passwordFields.forEach(field => {
            if (field.value && !this.validatePassword(field)) {
                isValid = false;
            }
        });

        return isValid;
    }

    updateInputState(input, isValid) {
        if (isValid) {
            input.classList.remove('error');
            input.classList.add('valid');
        } else {
            input.classList.remove('valid');
            input.classList.add('error');
        }
    }

    showFieldError(field, message) {
        this.clearFieldError(field);
        
        const errorDiv = document.createElement('div');
        errorDiv.className = 'field-error';
        errorDiv.textContent = message;
        errorDiv.style.color = '#ff0000';
        errorDiv.style.fontSize = '12px';
        errorDiv.style.marginTop = '4px';
        
        field.parentElement.appendChild(errorDiv);
        field.classList.add('error');
    }

    clearFieldError(field) {
        const existingError = field.parentElement.querySelector('.field-error');
        if (existingError) {
            existingError.remove();
        }
        field.classList.remove('error');
    }

    showLoadingOverlay() {
        const overlay = document.getElementById('loadingOverlay');
        if (overlay) {
            overlay.classList.remove('hidden');
        }
    }

    hideLoadingOverlay() {
        const overlay = document.getElementById('loadingOverlay');
        if (overlay) {
            overlay.classList.add('hidden');
        }
    }

    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `spotify-notification spotify-alert-${type}`;
        notification.textContent = message;
        
        // Style the notification
        notification.style.position = 'fixed';
        notification.style.top = '20px';
        notification.style.right = '20px';
        notification.style.zIndex = '10000';
        notification.style.padding = '16px 24px';
        notification.style.borderRadius = '8px';
        notification.style.fontWeight = '500';
        notification.style.maxWidth = '300px';
        notification.style.boxShadow = '0 4px 12px rgba(0, 0, 0, 0.3)';
        
        document.body.appendChild(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            notification.remove();
        }, 5000);
    }

    // Utility methods
    formatCurrency(amount, currency = 'USD') {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: currency
        }).format(amount);
    }

    formatDate(date) {
        return new Intl.DateTimeFormat('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        }).format(new Date(date));
    }

    // API methods
    async makeApiCall(url, options = {}) {
        try {
            const response = await fetch(url, {
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    ...options.headers
                },
                ...options
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            return await response.json();
        } catch (error) {
            console.error('API call failed:', error);
            this.showNotification('An error occurred. Please try again.', 'error');
            throw error;
        }
    }
}

// Initialize the app when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new SpotifyApp();
});

// Export for use in other scripts
window.SpotifyApp = SpotifyApp;

